﻿Public Class FormMain

#Region "Definizione delle variabili"

    Private MaxImApp As New MaxIm.Application
    Private MaxImCam As New MaxIm.CCDCamera
    Private AscomTelescope As ASCOM.DriverAccess.Telescope
    Private TelescopeConnected As Boolean = False
    Private AscomUtil As New ASCOM.Utilities.Util
    Private AscomAstroUtils As New ASCOM.Astrometry.AstroUtils.AstroUtils
    Private AscomTransform As New ASCOM.Astrometry.Transform.Transform
    Private AscomNovas As New ASCOM.Astrometry.NOVAS.NOVAS31
    Private TelescopeName As String
    Private AbortImaging As Boolean
    Private StarCenter As Point
    Private CameraBitmap As Bitmap
    Private CameraGraph As Graphics
    Private StarPositionMinimumError As Double = 2.5
    Private StarPositionMaximumError As Double = 5
    Private FirstCentroid As PointF
    Private FirstTime As Double
    Private ElapsedTime As Double
    Private DecDrift As Double
    Private StarPosition As PointF
    Private DriftDraw As Boolean = False
    Public Shared PlateAngle As Double
    Private PlateScale As Double
    Private LastPierSide As ASCOM.DeviceInterface.PierSide
    Private Correction As Double
    Private SingleCorrection As Double
    Private CorrectionRepeat As Integer

#End Region

#Region "Impostazioni dell'applicazione"

    Private Sub FormMain_Load(sender As Object, e As EventArgs) Handles MyBase.Load
        REM ******************************************************
        REM * Carica le impostazioni all'avvio dell'applicazione *
        REM ******************************************************
        Me.Text = "AstroGiGi " & My.Application.Info.Version.ToString
        Me.Top = Val(GetSetting("AstroGiGi", "FormMain", "Top", "0"))
        Me.Left = Val(GetSetting("AstroGiGi", "FormMain", "Left", "0"))
        Me.Height = Val(GetSetting("AstroGiGi", "FormMain", "Height", "600"))
        Me.Width = Val(GetSetting("AstroGiGi", "FormMain", "Width", "800"))
        Me.WindowState = Val(GetSetting("AstroGiGi", "FormMain", "WindowState", "0"))
        TelescopeName = GetSetting("AstroGiGi", "ASCOM", "Telescope ID", "")
        NumericExpTime.Value = Val(GetSetting("AstroGiGi", "Camera", "ExpTime", "1"))
        ButtonConnect.Enabled = Not (TelescopeName = "")
        Select Case GetSetting("AstroGiGi", "Telescope", "Star Az", "S")
            Case "S"
                RadioButtonS.Checked = True
            Case "W"
                RadioButtonW.Checked = True
            Case "E"
                RadioButtonE.Checked = True
            Case Else
                RadioButtonS.Checked = True
        End Select
        PlateAngle = Val(GetSetting("AstroGiGi", "Last Data", "Position Angle", "0"))
        PlateScale = Val(GetSetting("AstroGiGi", "Last Data", "Image Scale", "0"))
        NumericCrop.Value = Val(GetSetting("AstroGiGi", "Last Data", "Crop", "480"))
        If PlateScale > 0 Then
            LabelPlateAngle.Text = "Angolo di rotazione: " & Format(PlateAngle, "0.00") & "°"
            LabelPlateScale.Text = "Scala: " & Format(PlateScale, "0.00") & " arcsec/px"
        Else
            LabelPlateAngle.Text = "Angolo di rotazione: N/D"
            LabelPlateScale.Text = "Scala: N/D"
        End If
        TimerStatus.Enabled = True
    End Sub

    Private Sub FormMain_FormClosed(sender As Object, e As FormClosedEventArgs) Handles Me.FormClosed
        REM **************************************************************
        REM * Salva le impostazioni attuali all'uscita dall'applicazione *
        REM **************************************************************
        TimerStatus.Enabled = False
        If Me.WindowState = FormWindowState.Minimized Then Me.WindowState = FormWindowState.Normal
        SaveSetting("AstroGiGi", "FormMain", "WindowState", CStr(Me.WindowState))
        Me.WindowState = FormWindowState.Normal
        SaveSetting("AstroGiGi", "FormMain", "Top", Me.Top.ToString)
        SaveSetting("AstroGiGi", "FormMain", "Left", Me.Left.ToString)
        SaveSetting("AstroGiGi", "FormMain", "Height", Me.Height.ToString)
        SaveSetting("AstroGiGi", "FormMain", "Width", Me.Width.ToString)
        SaveSetting("AstroGiGi", "Camera", "ExpTime", NumericExpTime.Value.ToString)
        SaveSetting("AstroGiGi", "Camera", "Binning", NumericBinning.Value.ToString)
        SaveSetting("AstroGiGi", "Last Data", "Position Angle", PlateAngle.ToString)
        SaveSetting("AstroGiGi", "Last Data", "Image Scale", PlateScale.ToString)
        SaveSetting("AstroGiGi", "Last Data", "Crop", NumericCrop.Value.ToString)
        If RadioButtonS.Checked Then SaveSetting("AstroGiGi", "Telescope", "Star Az", "S")
        If RadioButtonW.Checked Then SaveSetting("AstroGiGi", "Telescope", "Star Az", "W")
        If RadioButtonE.Checked Then SaveSetting("AstroGiGi", "Telescope", "Star Az", "E")
    End Sub

#End Region

#Region "Controllo del telescopio"

    Private Sub ButtonTelescopeSelect_Click(sender As Object, e As EventArgs) Handles ButtonTelescopeSelect.Click
        REM ********************************************
        REM * Seleziona il driver ASCOM del telescopio *
        REM ********************************************
        TelescopeName = ASCOM.DriverAccess.Telescope.Choose(TelescopeName)
        SaveSetting("AstroGiGi", "ASCOM", "Telescope ID", TelescopeName)
        ButtonConnect.Enabled = Not (TelescopeName = "")
    End Sub

    'Private Sub ButtonTelescopeConnect_Click(sender As Object, e As EventArgs)
    '    REM ****************************************
    '    REM * Connetti o disconnetti il telescopio *
    '    REM ****************************************
    '    If ButtonTelescopeSelect.Visible Then
    '        ButtonTelescopeConnect.Image = ImageListLed.Images("LedYellow")
    '        AscomTelescope = New ASCOM.DriverAccess.Telescope(TelescopeName)
    '        Try
    '            AscomTelescope.Connected = True
    '            MaxImApp.TelescopeConnected = True
    '        Catch ex As Exception
    '            ButtonTelescopeConnect.Image = ImageListLed.Images("LedRed")
    '            Exit Sub
    '        End Try
    '        LastPierSide = AscomTelescope.SideOfPier
    '        ButtonTelescopeSelect.Visible = False
    '        ButtonTelescopeGo.Visible = True
    '        AscomTransform.SiteLatitude = AscomTelescope.SiteLatitude
    '        AscomTransform.SiteLongitude = AscomTelescope.SiteLongitude
    '        AscomTransform.SiteElevation = AscomTelescope.SiteElevation
    '        AscomTransform.SiteTemperature = 0
    '        ButtonTelescopeConnect.Image = ImageListLed.Images("LedGreen")
    '        AscomTelescope.Unpark()
    '        AscomTelescope.Tracking = True
    '        If Strings.Right(LabelDrift.Text, 3) <> "N/D" Then ButtonDrift.Enabled = True
    '    Else
    '        ButtonTelescopeConnect.Image = ImageListLed.Images("LedYellow")
    '        AscomUtil.WaitForMilliseconds(500)
    '        MaxImApp.TelescopeConnected = False
    '        AscomTelescope.Connected = False
    '        AscomTelescope.Dispose()
    '        AscomTelescope = Nothing
    '        ButtonTelescopeConnect.Image = ImageListLed.Images("LedBlack")
    '        ButtonTelescopeSelect.Visible = True
    '        ButtonTelescopeGo.Visible = False
    '        ButtonDrift.Enabled = False
    '    End If
    'End Sub

    Private Sub TimerStatus_Tick(sender As Object, e As EventArgs) Handles TimerStatus.Tick
        REM ************************************
        REM * Visualizza i dati del telescopio *
        REM ************************************
        Dim CameraStatus As MaxIm.CameraStatusCode
        Try
            CameraStatus = MaxImCam.CameraStatus
        Catch ex As Exception
            CameraStatus = MaxIm.CameraStatusCode.csError
        End Try
        Select Case CameraStatus
            Case MaxIm.CameraStatusCode.csIdle
                LabelCameraStatus.Text = "Status: Camera pronta"
                ButtonConnect.Image = ImageListLed.Images("LedGreen")
            Case MaxIm.CameraStatusCode.csNoCamera, MaxIm.CameraStatusCode.csError
                LabelCameraStatus.Text = "Status: Camera non connessa"
                ButtonConnect.Image = ImageListLed.Images("LedBlack")
            Case MaxIm.CameraStatusCode.csExposing
                LabelCameraStatus.Text = "Status: Esposizione in corso"
                ButtonConnect.Image = ImageListLed.Images("LedRed")
            Case MaxIm.CameraStatusCode.csExposingBias
                LabelCameraStatus.Text = "Status: Acquisizione di un Bias in corso"
                ButtonConnect.Image = ImageListLed.Images("LedRed")
            Case MaxIm.CameraStatusCode.csExposingDark
                LabelCameraStatus.Text = "Status: Acquisizione di un Dark in corso"
                ButtonConnect.Image = ImageListLed.Images("LedRed")
            Case MaxIm.CameraStatusCode.csReading
                LabelCameraStatus.Text = "Status: Lettura del CCD"
                ButtonConnect.Image = ImageListLed.Images("LedYellow")
            Case MaxIm.CameraStatusCode.csDownloading
                LabelCameraStatus.Text = "Status: Ricezione dei dati"
                ButtonConnect.Image = ImageListLed.Images("LedYellow")
            Case MaxIm.CameraStatusCode.csFlushing
                LabelCameraStatus.Text = "Status: Svuotamento delle cariche del CCD"
                ButtonConnect.Image = ImageListLed.Images("LedYellow")
            Case MaxIm.CameraStatusCode.csFilterWheelMoving
                LabelCameraStatus.Text = "Status: Rotazione della ruota portafiltri"
                ButtonConnect.Image = ImageListLed.Images("LedYellow")
            Case MaxIm.CameraStatusCode.csDelay
                LabelCameraStatus.Text = "Status: In pausa"
                ButtonConnect.Image = ImageListLed.Images("LedYellow")
            Case MaxIm.CameraStatusCode.csWaiting
                LabelCameraStatus.Text = "Status: In attesa"
                ButtonConnect.Image = ImageListLed.Images("LedYellow")
            Case MaxIm.CameraStatusCode.csCameraBusy
                LabelCameraStatus.Text = "Status: Camera occupata"
                ButtonConnect.Image = ImageListLed.Images("LedYellow")
            Case Else
                LabelCameraStatus.Text = "Status: " & CameraStatus.ToString
                ButtonConnect.Image = ImageListLed.Images("LedBlue")
        End Select
        Try
            Dim Temperature As Double = MaxImCam.Temperature
            If Temperature > 100 Or Temperature < -100 Then Temperature = 0
            LabelTemperature.Text = "Temperatura: " & Format(Temperature, "0.00") & "° Celsius"
        Catch ex As Exception
            LabelTemperature.Text = "Temperatura: N/D"
        End Try
        Try
            LabelPeltier.Text = "Cella di Peltier: " & Format(MaxImCam.CoolerPower, "0") & "%"
        Catch ex As Exception
            LabelPeltier.Text = "Cella di Peltier: N/D"
        End Try
        If TelescopeConnected Then
            LabelScopeRa.Text = "Ascensione retta: " & AscomUtil.HoursToHMS(AscomTelescope.RightAscension, ":", ":", "", 2)
            LabelScopeDec.Text = "Declinazione: " & AscomUtil.DegreesToDMS(AscomTelescope.Declination, ":", ":", "", 2)
            LabelScopeAz.Text = "Azimut: " & AscomUtil.DegreesToDMS(AscomTelescope.Azimuth, ":", ":", "", 2)
            LabelScopeAlt.Text = "Altezza: " & AscomUtil.DegreesToDMS(AscomTelescope.Altitude, ":", ":", "", 2)
            Select Case AscomTelescope.SideOfPier
                Case ASCOM.DeviceInterface.PierSide.pierEast
                    LabelScopePierSide.Text = "Meridiano: Normale"
                Case ASCOM.DeviceInterface.PierSide.pierWest
                    LabelScopePierSide.Text = "Meridiano: Inverso"
                Case Else
                    LabelScopePierSide.Text = "Meridiano: N/D"
            End Select
            If AscomTelescope.SideOfPier <> LastPierSide Then
                Dim A As Double = PlateAngle + 180
                If A >= 360 Then A = A - 360
                PlateAngle = A
                LabelPlateAngle.Text = "Angolo di rotazione: " & Format(PlateAngle, "0.00") & "°"
                LastPierSide = AscomTelescope.SideOfPier
            End If
            LabelSideralTime.Text = "Ora siderale: " & AscomUtil.HoursToHMS(AscomTelescope.SiderealTime, ":", ":", "", 2)
            Dim PointingError As Double
            If RadioButtonS.Checked Then
                PointingError = (AscomAstroUtils.ConditionHA(AscomTelescope.SiderealTime - AscomTelescope.RightAscension)) * 15
            End If
            If RadioButtonE.Checked Then
                PointingError = (AscomAstroUtils.ConditionHA(AscomTelescope.SiderealTime - AscomTelescope.RightAscension + 6)) * 15
            End If
            If RadioButtonW.Checked Then
                PointingError = (AscomAstroUtils.ConditionHA(AscomTelescope.SiderealTime - AscomTelescope.RightAscension - 6)) * 15
            End If
            If PointingError < 0 Then
                LabelPointingError.Text = "Errore: " & Format(Math.Abs(PointingError), "0.00") & "° verso Est"
            Else
                LabelPointingError.Text = "Errore: " & Format(Math.Abs(PointingError), "0.00") & "° verso Ovest"
            End If
            If AscomTelescope.Slewing Then
                ButtonTelescopeGo.Image = ImageListLed.Images("LedYellow")
            Else
                If Math.Abs(PointingError) > 1 Then
                    ButtonTelescopeGo.Image = ImageListLed.Images("LedRed")
                Else
                    ButtonTelescopeGo.Image = ImageListLed.Images("LedGreen")
                End If
            End If
        Else
            LabelScopeRa.Text = "Ascensione retta: N/D"
            LabelScopeDec.Text = "Declinazione: N/D"
            LabelScopeAz.Text = "Azimut: N/D"
            LabelScopeAlt.Text = "Altezza: N/D"
            LabelScopePierSide.Text = "Meridiano: N/D"
            LabelSideralTime.Text = "Ora siderale: N/D"
            LabelPointingError.Text = "Errore: N/D"
        End If
    End Sub

    Private Sub ButtonTelescopeGo_Click(sender As Object, e As EventArgs) Handles ButtonTelescopeGo.Click
        REM ***************************************************************
        REM * Punta il telescopio in direzione della stella da analizzare *
        REM ***************************************************************
        If AscomTelescope.AtPark Then AscomTelescope.Unpark()
        AscomTelescope.Tracking = True
        If RadioButtonS.Checked Then
            AscomTransform.SetAzimuthElevation(180, 45)
            AscomTelescope.SlewToCoordinatesAsync(AscomTransform.RAApparent, AscomTransform.DECApparent)
        End If
        If RadioButtonE.Checked Then
            AscomTelescope.SlewToCoordinatesAsync(AscomAstroUtils.ConditionRA(AscomTelescope.SiderealTime + 6), 45)
        End If
        If RadioButtonW.Checked Then
            AscomTelescope.SlewToCoordinatesAsync(AscomAstroUtils.ConditionRA(AscomTelescope.SiderealTime - 6), 45)
        End If
    End Sub

#End Region

#Region "Controllo della camera CCD"

    Private Sub NumericBinning_ValueChanged(sender As Object, e As EventArgs) Handles NumericBinning.ValueChanged
        REM *******************************************************************
        REM * Modifica il valore massimo consentito per il crop dell'immagine *
        REM *******************************************************************
        Dim CropMax As Integer = MaxImCam.CameraYSize / NumericBinning.Value
        If CropMax >= 480 Then
            NumericCrop.Enabled = True
            NumericCrop.Maximum = CropMax
        Else
            NumericCrop.Enabled = False
        End If
    End Sub

    Private Sub CheckBoxCooler_CheckedChanged(sender As Object, e As EventArgs) Handles CheckBoxCooler.CheckedChanged
        REM **************************************
        REM * Accendi o spegni il raffreddamento *
        REM **************************************
        If CheckBoxCooler.Enabled Then MaxImCam.CoolerOn = CheckBoxCooler.Checked
    End Sub

    Private Sub NumericSetPoint_ValueChanged(sender As Object, e As EventArgs) Handles NumericSetPoint.ValueChanged
        REM **************************************
        REM * Regola la temperatura della camera *
        REM **************************************
        If NumericSetPoint.Enabled Then MaxImCam.TemperatureSetpoint = NumericSetPoint.Value
    End Sub

    Private Sub ButtonConnect_Click(sender As Object, e As EventArgs) Handles ButtonConnect.Click
        REM ************************************
        REM * Connetti o disconnetti la camera *
        REM ************************************
        If MaxImCam.LinkEnabled Then
            If ButtonCancel.Enabled Then Exit Sub
            AscomTelescope.Tracking = False
            MaxImCam.LinkEnabled = False
            MaxImApp.TelescopeConnected = False
            SaveSetting("AstroGiGi", "Camera", "ExpTime", NumericExpTime.Value.ToString)
            SaveSetting("AstroGiGi", "Camera", "Binning", NumericBinning.Value.ToString)
            GroupBoxTelescope.Enabled = False
            GroupBoxCamera.Enabled = False
            GroupBoxPlateSolving.Enabled = False
            GroupBoxAnalysis.Enabled = False
            AscomTelescope.Dispose()
            AscomTelescope = Nothing
            TelescopeConnected = False
            ButtonTelescopeSelect.Enabled = True
        Else
            Try
                MaxImCam.LinkEnabled = True
                AscomTelescope = New ASCOM.DriverAccess.Telescope(TelescopeName)
                AscomTelescope.Connected = True
                MaxImApp.TelescopeConnected = True
                AscomTransform.SiteLatitude = AscomTelescope.SiteLatitude
                AscomTransform.SiteLongitude = AscomTelescope.SiteLongitude
                AscomTransform.SiteElevation = AscomTelescope.SiteElevation
                AscomTransform.SiteTemperature = 0
                AscomTelescope.Unpark()
                AscomTelescope.Tracking = True
                TelescopeConnected = True
                LastPierSide = AscomTelescope.SideOfPier
                Try
                    NumericBinning.Value = Val(GetSetting("AstroGiGi", "Camera", "Binning", "1"))
                Catch ex As Exception
                    NumericBinning.Value = 1
                End Try
                ComboBoxFilter.Items.Clear()
                If MaxImCam.HasFilterWheel Then
                    For Each FilterName In MaxImCam.FilterNames
                        ComboBoxFilter.Items.Add(FilterName)
                    Next
                    Try
                        ComboBoxFilter.SelectedIndex = MaxImCam.Filter
                    Catch ex As Exception
                        ComboBoxFilter.SelectedIndex = 0
                    End Try
                    ComboBoxFilter.Enabled = True
                Else
                    ComboBoxFilter.Enabled = False
                End If
                Try
                    NumericSetPoint.Value = MaxImCam.TemperatureSetpoint
                    NumericSetPoint.Enabled = True
                Catch ex As Exception
                    NumericSetPoint.Enabled = False
                End Try
                Try
                    CheckBoxCooler.Checked = MaxImCam.CoolerOn
                    CheckBoxCooler.Enabled = True
                Catch ex As Exception
                    CheckBoxCooler.Enabled = False
                End Try
                Dim CropMax As Integer = MaxImCam.CameraYSize / NumericBinning.Value
                If CropMax >= 480 Then
                    NumericCrop.Enabled = True
                    NumericCrop.Maximum = CropMax
                End If
                NumericBinning.Enabled = True
                NumericExpTime.Enabled = True
                NumericCrop.Enabled = True
                ButtonTake.Enabled = True
                ButtonTakeLoop.Enabled = True
                ButtonTelescopeSelect.Enabled = False
                GroupBoxTelescope.Enabled = True
                GroupBoxCamera.Enabled = True
                GroupBoxPlateSolving.Enabled = True
                GroupBoxAnalysis.Enabled = True
            Catch ex As Exception
                MaxImCam.LinkEnabled = False
                MaxImApp.TelescopeConnected = False
                AscomTelescope.Connected = False
                GroupBoxTelescope.Enabled = False
                GroupBoxCamera.Enabled = False
                GroupBoxPlateSolving.Enabled = False
                GroupBoxAnalysis.Enabled = False
                TelescopeConnected = False
                MsgBox(Err.Description, MsgBoxStyle.Critical, "Errore " & Err.Number.ToString)
            End Try
        End If
    End Sub

    Private Sub ButtonTake_Click(sender As Object, e As EventArgs) Handles ButtonTake.Click
        REM ***********************************
        REM * Acquisisci una singola immagine *
        REM ***********************************
        MaxImCam.BinX = NumericBinning.Value
        MaxImCam.BinY = NumericBinning.Value
        MaxImCam.SetFullFrame()
        If MaxImCam.NumX < 480 Or MaxImCam.NumY < 480 Then
            MsgBox("AstroGiGi non può funzionare se la dimensione dell'immagine non è almeno di 480x480 pixel", MsgBoxStyle.Critical + MsgBoxStyle.OkOnly, "Errore!")
            Exit Sub
        End If
        MaxImCam.StartX = (MaxImCam.CameraXSize / NumericBinning.Value - NumericCrop.Value) / 2
        MaxImCam.StartY = (MaxImCam.CameraYSize / NumericBinning.Value - NumericCrop.Value) / 2
        MaxImCam.NumX = NumericCrop.Value
        MaxImCam.NumY = NumericCrop.Value
        MaxImApp.CloseAll()
        AbortImaging = False
        ButtonCancel.Enabled = True
        ButtonTake.Enabled = False
        ButtonTakeLoop.Enabled = False
        NumericExpTime.Enabled = False
        NumericCrop.Enabled = False
        NumericBinning.Enabled = False
        ComboBoxFilter.Enabled = False
        PictureBoxCamera.Cursor = Cursors.WaitCursor
        Me.Cursor = Cursors.WaitCursor
        TakeImage(NumericExpTime.Value, NumericBinning.Value)
        Me.Cursor = Cursors.Default
        PictureBoxCamera.Cursor = Cursors.Cross
        NumericExpTime.Enabled = True
        NumericCrop.Enabled = True
        NumericBinning.Enabled = True
        ComboBoxFilter.Enabled = True
        ButtonCancel.Enabled = False
        ButtonTake.Enabled = True
        ButtonTakeLoop.Enabled = True
        PictureBoxCamera.Enabled = True
    End Sub

    Private Sub ButtonTakeLoop_Click(sender As Object, e As EventArgs) Handles ButtonTakeLoop.Click
        REM ************************************
        REM * Acquisisci una serie di immagini *
        REM ************************************
        MaxImCam.BinX = NumericBinning.Value
        MaxImCam.BinY = NumericBinning.Value
        MaxImCam.SetFullFrame()
        If MaxImCam.NumX < 480 Or MaxImCam.NumY < 480 Then
            MsgBox("AstroGiGi non può funzionare se la dimensione dell'immagine non è almeno di 480x480 pixel", MsgBoxStyle.Critical + MsgBoxStyle.OkOnly, "Errore!")
            Exit Sub
        End If
        MaxImCam.StartX = (MaxImCam.CameraXSize / NumericBinning.Value - NumericCrop.Value) / 2
        MaxImCam.StartY = (MaxImCam.CameraYSize / NumericBinning.Value - NumericCrop.Value) / 2
        MaxImCam.NumX = NumericCrop.Value
        MaxImCam.NumY = NumericCrop.Value
        MaxImApp.CloseAll()
        AbortImaging = False
        ButtonTake.Enabled = False
        ButtonTakeLoop.Enabled = False
        ButtonCancel.Enabled = True
        NumericExpTime.Enabled = False
        NumericCrop.Enabled = False
        NumericBinning.Enabled = False
        ComboBoxFilter.Enabled = False
        Me.Cursor = Cursors.WaitCursor
        PictureBoxCamera.Cursor = Cursors.WaitCursor
        TimerLoop.Enabled = True
    End Sub

    Private Sub PictureBoxCamera_MouseUp(sender As Object, e As MouseEventArgs) Handles PictureBoxCamera.MouseUp
        REM **********************************
        REM * Gestisci i click sull'immagine *
        REM **********************************
        If ButtonCancel.Enabled Then Exit Sub
        If e.X < 127 Or e.X > PictureBoxCamera.Width - 128 Or e.Y < 127 Or e.Y > PictureBoxCamera.Height - 128 Then
            MsgBox("La stella selezionata si trova troppo vicino al bordo dell'immagine", MsgBoxStyle.OkOnly + MsgBoxStyle.Critical, "Operazione annullata!")
            Exit Sub
        End If
        DriftDraw = False
        ButtonDrift.Image = ImageListLed.Images("LedBlack")
        ButtonDrift.Enabled = False
        MaxImApp.CloseAll()
        PictureBoxCamera.Enabled = False
        AbortImaging = False
        ButtonTake.Enabled = False
        ButtonTakeLoop.Enabled = False
        ButtonCancel.Enabled = True
        NumericExpTime.Enabled = False
        NumericCrop.Enabled = False
        NumericBinning.Enabled = False
        ComboBoxFilter.Enabled = False
        StarCenter.X = e.Location.X + (MaxImCam.CameraXSize / MaxImCam.BinX - NumericCrop.Value) / 2
        StarCenter.Y = e.Location.Y + (MaxImCam.CameraYSize / MaxImCam.BinY - NumericCrop.Value) / 2
        AbortImaging = False
        Me.Cursor = Cursors.WaitCursor
        TimerStar.Enabled = True
    End Sub

    Private Sub ButtonCancel_Click(sender As Object, e As EventArgs) Handles ButtonCancel.Click
        REM ********************************************
        REM * Interrompi l'acquisizione delle immagini *
        REM ********************************************
        ButtonCancel.Enabled = False
        AbortImaging = True
    End Sub

    Private Sub TakeImage(ExpTime As Double, Binning As Short)
        REM *****************************************
        REM * Acquisisci un'immagine e visualizzala *
        REM *****************************************
        MaxImCam.BinX = Binning
        MaxImCam.BinY = Binning
        If MaxImCam.HasFilterWheel Then
            MaxImCam.Expose(ExpTime, 1, ComboBoxFilter.SelectedIndex)
        Else
            MaxImCam.Expose(ExpTime, 1)
        End If
        Do
            AscomUtil.WaitForMilliseconds(50)
            If AbortImaging Then
                MaxImCam.AbortExposure()
                Exit Sub
            End If
        Loop Until MaxImCam.ImageReady
        Do
            AscomUtil.WaitForMilliseconds(50)
        Loop Until MaxImCam.CameraStatus = MaxIm.CameraStatusCode.csIdle
        AscomUtil.WaitForMilliseconds(50)
        ShowMaxImDoc(MaxImApp.FirstDocument)
        CameraGraph = Graphics.FromImage(CameraBitmap)
        If DriftDraw Then
            StarPosition.X = MaxImApp.FirstDocument.XSize / 2
            StarPosition.Y = MaxImApp.FirstDocument.YSize / 2
            Circle(StarPosition, Pens.Red)
            Dim DestinationCenter As PointF
            If RadioButtonS.Checked Then
                DestinationCenter.X = StarPosition.X + DistanceX(SingleCorrection)
                DestinationCenter.Y = StarPosition.Y + DistanceY(SingleCorrection)
            Else
                DestinationCenter.X = StarPosition.X - DistanceX(SingleCorrection)
                DestinationCenter.Y = StarPosition.Y - DistanceY(SingleCorrection)
            End If
            Circle(DestinationCenter, Pens.LightGreen)
            Dim StartLinePosition As PointF
            StartLinePosition.X = StarPosition.X
            StartLinePosition.Y = StarPosition.Y
            Dim EndLinePosition As PointF
            EndLinePosition.X = DestinationCenter.X
            EndLinePosition.Y = DestinationCenter.Y
            CameraGraph.SmoothingMode = Drawing2D.SmoothingMode.AntiAlias
            CameraGraph.DrawLine(Pens.Yellow, StartLinePosition, EndLinePosition)
            CameraGraph.SmoothingMode = Drawing2D.SmoothingMode.Default
            Dim CameraFont As New Font("Arial", 32, FontStyle.Bold)
            Dim CameraBrush As New SolidBrush(Color.Magenta)
            CameraGraph.DrawString(CorrectionRepeat.ToString & "x", CameraFont, CameraBrush, 10, 10)
        Else
            CameraGraph.DrawRectangle(Pens.Red, 127, 127, MaxImCam.NumX - 255, MaxImCam.NumY - 255)
        End If
        PictureBoxCamera.Image = CameraBitmap
    End Sub

    Private Sub TimerLoop_Tick(sender As Object, e As EventArgs) Handles TimerLoop.Tick
        REM ************************************************
        REM * Acquisisci una serie di immagini in sequenza *
        REM ************************************************
        TimerLoop.Enabled = False
        Do Until AbortImaging
            TakeImage(NumericExpTime.Value, NumericBinning.Value)
            My.Application.DoEvents()
            AscomUtil.WaitForMilliseconds(100)
        Loop
        Me.Cursor = Cursors.Default
        PictureBoxCamera.Cursor = Cursors.Cross
        NumericExpTime.Enabled = True
        NumericCrop.Enabled = True
        NumericBinning.Enabled = True
        ComboBoxFilter.Enabled = True
        ButtonCancel.Enabled = False
        ButtonTake.Enabled = True
        ButtonTakeLoop.Enabled = True
        PictureBoxCamera.Enabled = True
    End Sub

    Private Sub TimerStar_Tick(sender As Object, e As EventArgs) Handles TimerStar.Tick
        REM *****************************************************************
        REM * Esegui un ciclo di riprese e misura la posizione della stella *
        REM *****************************************************************
        TimerStar.Enabled = False
        ButtonReset.Enabled = True
        FirstCentroid.X = 0
        FirstCentroid.Y = 0
        MaxImCam.StartX = StarCenter.X - 127
        MaxImCam.StartY = StarCenter.Y - 127
        MaxImCam.NumX = 255
        MaxImCam.NumY = 255
        Dim SubFrameStarCenter As Point
        SubFrameStarCenter.X = 127
        SubFrameStarCenter.Y = 127
        If Not AbortImaging Then
            Dim ImageInfo
            Do Until AbortImaging
                TakeImage(NumericExpTime.Value, NumericBinning.Value)
                Dim Ring() As Integer = {40, 5, 5}
                ImageInfo = MaxImApp.FirstDocument.CalcInformation(SubFrameStarCenter.X, SubFrameStarCenter.Y, Ring)
                If FirstCentroid.X = 0 And FirstCentroid.Y = 0 Then
                    FirstTime = My.Computer.Clock.TickCount
                    FirstCentroid.X = ImageInfo(6)
                    FirstCentroid.Y = ImageInfo(7)
                Else
                    Dim Drift As PointF
                    ElapsedTime = (My.Computer.Clock.TickCount - FirstTime) / 1000
                    LabelTime.Text = "Tempo: " & Format(ElapsedTime, "0.0") & " sec"
                    Drift.X = ImageInfo(6) - FirstCentroid.X
                    Drift.Y = ImageInfo(7) - FirstCentroid.Y
                    If ImageInfo(6) < 40 Or ImageInfo(6) > 215 Or ImageInfo(7) < 40 Or ImageInfo(7) > 215 Then
                        MsgBox("La stella si è avvicinata troppo al bordo dell'immagine", MsgBoxStyle.Exclamation + MsgBoxStyle.OkOnly, "Analisi interrotta")
                        Exit Do
                    End If
                    DecDrift = Rotate(Drift, PlateAngle).Y
                    LabelDrift.Text = "Deriva: " & Format(DecDrift, "0.0000") & " px"
                    LabelDriftRate.Text = "Velocità: " & Format(DecDrift / ElapsedTime, "0.0000") & " px/sec"
                    If RadioButtonS.Checked Then
                        Correction = DecDrift * (Math.Cos(DegToRad(AscomTelescope.Altitude)) / Math.Cos(DegToRad(AscomTelescope.SiteLatitude))) / Math.Sin(2 * Math.PI * ElapsedTime / 86164)
                        LabelCorrection.Text = "Correzione: " & Format(Math.Round(Correction), "0") & " px"
                        If DecDrift < 0 Then
                            LabelDirection.Text = "Direzione: Azimut in senso orario"
                        Else
                            LabelDirection.Text = "Direzione: Azimut in senso antiorario"
                        End If
                    Else
                        Correction = DecDrift * Math.Sin(DegToRad(AscomTelescope.Declination)) / Math.Sin(2 * Math.PI * ElapsedTime / 86164)
                        LabelCorrection.Text = "Correzione: " & Format(Math.Round(Correction), "0") & " px"
                        If RadioButtonE.Checked Then
                            If DecDrift < 0 Then
                                LabelDirection.Text = "Direzione: Diminuire l'altezza"
                            Else
                                LabelDirection.Text = "Direzione: Aumentare l'altezza"
                            End If
                        Else
                            If DecDrift < 0 Then
                                LabelDirection.Text = "Direzione: Aumentare l'altezza"
                            Else
                                LabelDirection.Text = "Direzione: Diminuire l'altezza"
                            End If
                        End If
                    End If
                End If
                SubFrameStarCenter.X = Math.Round(ImageInfo(6))
                SubFrameStarCenter.Y = Math.Round(ImageInfo(7))
                CameraGraph = Graphics.FromImage(CameraBitmap)
                Dim N As Short
                For N = -1 To 1
                    CameraGraph.DrawLine(Pens.Red, SubFrameStarCenter.X - 40, SubFrameStarCenter.Y + N, SubFrameStarCenter.X - 10, SubFrameStarCenter.Y + N)
                    CameraGraph.DrawLine(Pens.Red, SubFrameStarCenter.X + N, SubFrameStarCenter.Y - 40, SubFrameStarCenter.X + N, SubFrameStarCenter.Y - 10)
                    CameraGraph.DrawLine(Pens.Red, SubFrameStarCenter.X + 40, SubFrameStarCenter.Y + N, SubFrameStarCenter.X + 10, SubFrameStarCenter.Y + N)
                    CameraGraph.DrawLine(Pens.Red, SubFrameStarCenter.X + N, SubFrameStarCenter.Y + 40, SubFrameStarCenter.X + N, SubFrameStarCenter.Y + 10)
                Next
                PictureBoxCamera.Image = CameraBitmap
                PictureBoxCamera.Refresh()
                My.Application.DoEvents()
                AscomUtil.WaitForMilliseconds(100)
            Loop
        End If
        AbortImaging = False
        MaxImCam.SetFullFrame()
        MaxImCam.StartX = (MaxImCam.CameraXSize / NumericBinning.Value - NumericCrop.Value) / 2
        MaxImCam.StartY = (MaxImCam.CameraYSize / NumericBinning.Value - NumericCrop.Value) / 2
        MaxImCam.NumX = NumericCrop.Value
        MaxImCam.NumY = NumericCrop.Value
        TakeImage(NumericExpTime.Value, NumericBinning.Value)
        ButtonReset.Enabled = False
        PictureBoxCamera.Enabled = True
        ButtonCancel.Enabled = False
        ButtonTake.Enabled = True
        ButtonTakeLoop.Enabled = True
        NumericExpTime.Enabled = True
        NumericCrop.Enabled = True
        NumericBinning.Enabled = True
        ComboBoxFilter.Enabled = True
        ButtonDrift.Enabled = True
        Me.Cursor = Cursors.Default
    End Sub

    Private Sub ButtonReset_Click(sender As Object, e As EventArgs) Handles ButtonReset.Click
        REM ****************************************
        REM * Azzera i contatori durante l'analisi *
        REM ****************************************
        FirstCentroid.X = 0
        FirstCentroid.Y = 0
    End Sub

    Public Sub ShowMaxImDoc(MaxImDoc As MaxIm.Document)
        REM *******************************
        REM * Visualizza un'immagine FITS *
        REM *******************************
        PictureBoxCamera.Width = MaxImDoc.XSize
        PictureBoxCamera.Height = MaxImDoc.YSize
        Dim BitmapArray() As Byte
        BitmapArray = MaxImDoc.StretchedBMP
        Dim BitmapLenght As Integer = BitmapArray.Length + 14
        Dim BitmapLenghtHexString As String = "00000000" & Hex(BitmapLenght)
        BitmapLenghtHexString = Strings.Right(BitmapLenghtHexString, 8)
        Dim BitmapFileHeader(13) As Byte
        BitmapFileHeader(0) = &H42
        BitmapFileHeader(1) = &H4D
        BitmapFileHeader(2) = Val("&H" & Mid(BitmapLenghtHexString, 7, 2))
        BitmapFileHeader(3) = Val("&H" & Mid(BitmapLenghtHexString, 5, 2))
        BitmapFileHeader(4) = Val("&H" & Mid(BitmapLenghtHexString, 3, 2))
        BitmapFileHeader(5) = Val("&H" & Mid(BitmapLenghtHexString, 1, 2))
        BitmapFileHeader(6) = &H0
        BitmapFileHeader(7) = &H0
        BitmapFileHeader(8) = &H0
        BitmapFileHeader(9) = &H0
        BitmapFileHeader(10) = &H36
        BitmapFileHeader(11) = &H0
        BitmapFileHeader(12) = &H0
        BitmapFileHeader(13) = &H0
        Dim BitmapImageArray(BitmapLenght) As Byte
        BitmapFileHeader.CopyTo(BitmapImageArray, 0)
        BitmapArray.CopyTo(BitmapImageArray, 14)
        Dim ImageConvert As New ImageConverter
        CameraBitmap = ImageConvert.ConvertFrom(BitmapImageArray)
        PictureBoxCamera.Image = CameraBitmap
    End Sub

#End Region

#Region "Orientamento della camera"

    Private Function Rotate(Centroid As PointF, Rotation As Double) As PointF
        REM *************************************************
        REM * Ruota due coordinate di un angolo specificato *
        REM *************************************************
        Dim OrigCoord() As Double = {Centroid.X, Centroid.Y, 0}
        Dim RotatedCoord() As Double = {0, 0, 0}
        AscomNovas.Spin(Rotation, OrigCoord, RotatedCoord)
        Dim Result As PointF
        Result.X = RotatedCoord(0)
        Result.Y = RotatedCoord(1)
        Return Result
    End Function

    Private Sub ButtonMaxImPlate_Click(sender As Object, e As EventArgs) Handles ButtonMaxImPlate.Click
        REM *************************************
        REM * Ricava l'orientamento da MaxIm DL *
        REM *************************************
        If MsgBox("Esegui il plate solving con MaxIm DL quindi clicca su OK", MsgBoxStyle.Information + MsgBoxStyle.OkCancel, "Orientamento della camera") = MsgBoxResult.Ok Then
            Dim S As Double
            Dim A As Double
            Try
                S = MaxImApp.FirstDocument.ImageScale
                A = MaxImApp.FirstDocument.PositionAngle
                Dim CD1 As Double = MaxImApp.FirstDocument.GetFITSKey("CDELT1")
                Dim CD2 As Double = MaxImApp.FirstDocument.GetFITSKey("CDELT2")
                If CD1 * CD2 < 0 Then
                    MsgBox("Modificare le impostazioni della camera in MaxIm DL", MsgBoxStyle.OkOnly + vbExclamation, "Immagine specchiata!")
                    Exit Sub
                End If

            Catch ex As Exception
                MsgBox("Non è stato possibile leggere i dati da MaxIm DL", MsgBoxStyle.Critical + MsgBoxStyle.OkOnly, "Errore!")
                Exit Sub
            End Try
            If S < 0 Then
                A = 360 - A
            Else
                A = 180 - A
            End If
            If A >= 360 Then A = A - 360
            If A < 0 Then A = A + 360
            PlateAngle = A
            PlateScale = Math.Abs(S)
            LabelPlateAngle.Text = "Angolo di rotazione: " & Format(PlateAngle, "0.00") & "°"
            LabelPlateScale.Text = "Scala: " & Format(PlateScale, "0.00") & " arcsec/px"
        End If
    End Sub

    Private Sub ButtonManualRotation_Click(sender As Object, e As EventArgs) Handles ButtonManualRotation.Click
        REM ************************************************************
        REM * Inserimento manuale dell'angolo di rotazione della camera*
        REM ************************************************************
        If DialogAngle.ShowDialog() = DialogResult.OK Then
            LabelPlateAngle.Text = "Angolo di rotazione: " & Format(PlateAngle, "0.00") & "°"
        End If
    End Sub

#End Region

#Region "Calcolo della deriva"

    Private Sub ButtonDrift_Click(sender As Object, e As EventArgs) Handles ButtonDrift.Click
        REM ***************************************************************************
        REM * Seleziona una stella di cui verrà mostrato lo spostamento da effettuare *
        REM ***************************************************************************
        If DriftDraw Then
            DriftDraw = False
            ButtonDrift.Image = ImageListLed.Images("LedYellow")
            ButtonTake.PerformClick()
            ButtonDrift.Image = ImageListLed.Images("LedBlack")
        Else
            ButtonDrift.Image = ImageListLed.Images("LedYellow")
            CorrectionRepeat = Int(Math.Abs(Correction) / (PictureBoxCamera.Height / 2 - 35)) + 1
            SingleCorrection = Correction / CorrectionRepeat
            DriftDraw = True
            ButtonTake.PerformClick()
            ButtonDrift.Image = ImageListLed.Images("LedGreen")
            ButtonTakeLoop.PerformClick()
        End If
    End Sub

    Private Function DistanceX(Distance As Double) As Single
        REM **********************************************************************************************************
        REM * Restituisci lo spostamento orizzontale di un punto ad una certa distanza sull'asse di Ascensione Retta *
        REM **********************************************************************************************************
        Return Distance * Math.Cos(DegToRad(PlateAngle))
    End Function

    Private Function DistanceY(Distance As Double) As Single
        REM ********************************************************************************************************
        REM * Restituisci lo spostamento verticale di un punto ad una certa distanza sull'asse di Ascensione Retta *
        REM ********************************************************************************************************
        Return Distance * Math.Sin(DegToRad(PlateAngle))
    End Function

    Public Function DegToRad(Degrees As Double) As Double
        REM **********************************
        REM * Converti un angolo in radianti *
        REM **********************************
        Return Degrees * Math.PI / 180
    End Function

    Private Sub Circle(CircleCenter As PointF, CircleColor As Pen)
        REM *******************************************************
        REM * Disegna un cerchio attorno alle coordinate indicate *
        REM *******************************************************
        CameraGraph.SmoothingMode = Drawing2D.SmoothingMode.AntiAlias
        CameraGraph.DrawEllipse(CircleColor, CSng(CircleCenter.X - 15), CSng(CircleCenter.Y - 15), 30, 30)
        CameraGraph.DrawEllipse(CircleColor, CSng(CircleCenter.X - 15.5), CSng(CircleCenter.Y - 15.5), 31, 31)
        CameraGraph.DrawEllipse(CircleColor, CSng(CircleCenter.X - 16), CSng(CircleCenter.Y - 16), 32, 32)
        CameraGraph.SmoothingMode = Drawing2D.SmoothingMode.Default
    End Sub

#End Region

End Class